%% Supplementary software for the article:
%
% "Photon count estimation in single-molecule localization microscopy"
% Rasmus ?. Thorsen, Christiaan N. Hulleman, Sjoerd Stallinga and Bernd Rieger
%
% This .m file reproduces the simulation part of Figure 1B. 
% - (Example 2) The photometric ratio 
% 1) Compute a through-focus PSF
% 2) Estimate photon count by Gaussian (MLE-)fit
% 3) Estimate photon count by TRABI
% 4) Compute photometric ratio (Gaussian fit over TRABI value)
%
% - figure 1
% through-focus (z) PSF(x,y)-stack
%
% - figure 2
% Estimated photon count by gaussian fit and trabi as a function of the 
% axial position, and the corresponding photometric ratio.

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands                     

%% Example 2 - initialize
clearvars
close all
clc

help example2

Nph = 10000; % Numbere of signal photons
Nbg = 10; % Number of background photons per pixel

%% vectorial PSF model
addpath('matlabfun/vector PSF') % add psf scripts

% set parameters (see script)
parameters = set_parameters;
% compute through focus PSF with specfied parameters
[XImage,YImage,ZImage,allPSFs] = get_vectorPSF(parameters);


norm_allPSFs = sum(sum(allPSFs,1),2); % normalization factor
tmp = size(allPSFs);
tmp = tmp(1:2);
norm_allPSFs = repmat(norm_allPSFs,[tmp,1]);
allPSFs = allPSFs./norm_allPSFs; % normalize psf
allspots = allPSFs*Nph+Nbg; % spot with constant background

rmpath('matlabfun/vector PSF') % remove psf scripts

%% Estimate number of photons

%%%% Gaussian fit
addpath('matlabfun/gaussian fitter') % add gaussian fitt scripts

ROI = 7; % set ROI [px]
theta_gaussian = get_gaussfitter(allspots,ROI,parameters); % output: estimated (x,y,z,sigma,Nph,Nbg)

rmpath('matlabfun/gaussian fitter') % remove gaussian fit scripts

%%%% photometry
addpath('matlabfun/trabi') % add trabi scripts

radius = 214.5*1.86/parameters.pixelsize; % set aperture radius [px]
theta_trabi = get_trabi(allspots,radius,Nbg,parameters); % output: estimated (Nph)

rmpath('matlabfun/trabi') % remove trabi scripts

%% Photometric ratio
P = theta_gaussian(:,4)./theta_trabi;

%% Plot

%%% Through-focus PSF
scrsz = [1 1 1366 768];
figure
set(gcf,'Position',[.5*scrsz(4) 0.7*scrsz(4) 1.0*scrsz(4) 0.3*scrsz(4)]);

dz = 0;
subplot(1,3,1)
imagesc(XImage(:,1),YImage(1,:),allspots(:,:,round(parameters.Mz/2)+dz)) % // add radial average 
ylabel('y [nm]')
xlabel('x [nm]')
ylim([-1000 1000])
xlim([-1000 1000])
title(['z = ' num2str(ZImage(round(parameters.Mz/2)+dz),3) ' [nm]'])
axis square

dz = 6;
subplot(1,3,2)
imagesc(XImage(:,1),YImage(1,:),allspots(:,:,round(parameters.Mz/2)+dz)) % // add radial average 
ylabel('y [nm]')
xlabel('x [nm]')
ylim([-1000 1000])
xlim([-1000 1000])
title(['z = ' num2str(ZImage(round(parameters.Mz/2)+dz),3) ' [nm]'])
axis square

dz = 12;
subplot(1,3,3)
imagesc(XImage(:,1),YImage(1,:),allspots(:,:,round(parameters.Mz/2)+dz)) % // add radial average 
ylabel('y [nm]')
xlabel('x [nm]')
ylim([-1000 1000])
xlim([-1000 1000])
title(['z = ' num2str(ZImage(round(parameters.Mz/2)+dz),3) ' [nm]'])
axis square 

%%% Photon count and photometric ratio
scrsz = [1 1 1366 768];
figure
set(gcf,'Position',[.5*scrsz(4) 0.25*scrsz(4) 1.0*scrsz(4) 0.3*scrsz(4)]);

subplot(1,3,1)
plot(ZImage,theta_gaussian(:,4))
ylabel('Estimated photon count')
xlabel('Axial position [nm]')
title('Gaussian fit')
axis square
ylim([0 Nph])
xlim([-600 600])

subplot(1,3,2)
plot(ZImage,theta_trabi)
ylabel('Estimated photon count')
xlabel('Axial position [nm]')
title('TRABI')
axis square
ylim([0 Nph])
xlim([-600 600])

subplot(1,3,3)
plot(ZImage,P*100)
ylabel('Photometric ratio [%]')
xlabel('Axial position')
title('Gaussian fit over TRABI')
axis square
xlim([-600 600])

